<?php

namespace App\Http\Controllers\Site;

use App\Http\Controllers\Controller;
use App\Models\Edu\DashboardPalette\DashboardPalette;
use App\Models\Modules\SubModule;
use App\Models\Permission\UserPermission;
use App\Models\Site\Role\SiteRole;
use App\Models\User\User;
use App\Traits\Site\UserPermission\UserAccessTrait;
use Exception;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use Illuminate\Validation\ValidationException;
use Response;
use Session;

class RolesController extends Controller
{

    use UserAccessTrait;

    private $role;
    private $user;
    private $dashboardPalette;
    private $subModule;
    private $userPermission;

    public function __construct(SiteRole $role, User $user, DashboardPalette $dashboardPalette, SubModule $subModule, UserPermission $userPermission)
    {
        $this->middleware('auth');
        $this->middleware('sitepagechecker');
        $this->role = $role;
        $this->user = $user;
        $this->dashboardPalette = $dashboardPalette;
        $this->subModule = $subModule;
        $this->userPermission = $userPermission;
    }


    public function create(Request $validatedRequest)
    {
        try {
            $this->validate($validatedRequest, [
                'name' => 'required|unique:site_roles,name,id,0,site_id,' . Session::get('SITE_ID'),
                'sub_module_ids' => 'required',
            ]);

            $this->role = new SiteRole();
            $sub_module_ids = implode(',', $validatedRequest->sub_module_ids) ?? null;

            if (count($this->role->ownFields) > 0) {
                foreach ($this->role->ownFields as $ownField) {
                    if ($validatedRequest->{$ownField}) {
                        $this->role->{$ownField} = $validatedRequest->{$ownField};
                    }
                }
                $this->role->key = 'general';
            }

            if ($this->role->save()) {
                $this->role->groupAccess()->create([
                    'site_role_id' => $this->role->id,
                    'site_id' => $this->role->site_id,
                    'sub_module_ids' => $sub_module_ids,
                ]);

                $this->saveDashboardPalette($validatedRequest->sub_module_ids, $this->role);
                return Response::json(['message' => 'Saved Successfully!', 'mode' => 'Saved!'], 200);

            }

        } catch (ValidationException $exception) {
            return JsonResponse::create(['message' => $exception->getMessage(), 'errors' => $exception->validator->getMessageBag()->toArray()], 422);
        } catch (Exception $exception) {
            return JsonResponse::create(['message' => $exception->getMessage()], 410);
        }

    }

    public function update(Request $validatedRequest, $id)
    {
        try {
            $this->validate($validatedRequest, [
                'name' => 'required|unique:site_roles,name,' . $id . ',id,site_id,' . Session::get('SITE_ID'),
                'sub_module_ids' => 'required',
            ]);
            $toEdit = $this->role->findOrFail($id);
            $sub_module_ids = implode(',', $validatedRequest->sub_module_ids) ?? null;

            // \Log::info($sub_module_ids);

            if (count($toEdit->ownFields) > 0) {
                foreach ($toEdit->ownFields as $ownField) {
                    if ($validatedRequest->{$ownField}) {
                        $toEdit->{$ownField} = $validatedRequest->{$ownField};
                    }
                }
            }

            if ($toEdit->groupAccess == null) {
                $toEdit->groupAccess()->create([
                    'site_role_id' => $toEdit->id,
                    'site_id' => $toEdit->site_id,
                    'sub_module_ids' => $sub_module_ids,
                ]);
            } else {
                $toEdit->groupAccess->update([
                    'sub_module_ids' => $sub_module_ids,
                ]);
            }

            $users = $toEdit->users()->pluck('users.id')->toArray();

            $this->rearrangeSidebar($id);

            if ($toEdit->save()) {

                $this->saveDashboardPalette($validatedRequest->sub_module_ids, $toEdit);
                $this->setUserPermission($users);

                return Response::json(['message' => 'Updated Successfully', 'mode' => 'Updated!'], 200);
            }
        } catch (ValidationException $exception) {
            return JsonResponse::create(['message' => $exception->getMessage(), 'errors' => $exception->validator->getMessageBag()->toArray()], 410);
        } catch (Exception $exception) {
            return JsonResponse::create(['message' => $exception->getMessage()], 410);
        }
    }

    private function saveDashboardPalette($sub_module_ids, $toEdit)
    {
        $dashboard_palette = $this->subModule
            ->whereIn('id', $sub_module_ids)
            ->whereNotNull('palette_id')
            ->pluck('palette_id')->toArray();

        $dashboard_palette_ids = implode(',', $dashboard_palette) ?? null;


        $toEdit->groupAccess->update([
            'dashboard_palettes' => $dashboard_palette_ids,
        ]);

        $toEdit->save();
    }


    public function delete($id)
    {
        try {

            $the_role = $this->role->findOrFail($id);

            if ($the_role->key == 'admin' || $the_role->key == 'employee' || $the_role->key == 'student' || $the_role->key == 'guardian') {
                throw new Exception('Can"t Delete Default Role!');
            }

            $the_role->delete();
            return Response::json(['message' => $the_role->name . ' Deleted Successfully', 'mode' => 'Delete!'], 200);
        } catch (Exception $exception) {
            return JsonResponse::create(['message' => $exception->getMessage()], 410);
        }
    }





    //CUSTOM METHOD---------------------------------


    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\JsonResponse
     */
    public function getRolesList()
    {
        try {
            $roles = $this->role
                ->all();
            return response()->json($roles, 200, []);
        } catch (Exception $exception) {
            return JsonResponse::create(['message' => $exception->getMessage()], 410);
        }

    }

    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\JsonResponse
     */

    public function getRolesWithAccesses()
    {
        try {
            $roles = $this->role
                ->with('accesses')
                ->get();
            return response()->json($roles, 200, []);
        } catch (Exception $exception) {
            return JsonResponse::create(['message' => $exception->getMessage()], 410);
        }
    }

    public function getPaletteList()
    {
        try {
            $palettes = $this->dashboardPalette
                ->get();
            return response()->json($palettes, 200);
        } catch (Exception $exception) {
            return JsonResponse::create(['message' => $exception->getMessage()], 410);
        }
    }

    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\JsonResponse
     */
    public function getRolesListAndUserAssignedRoles($user_id)
    {
        try {
            $the_user = $this->user
                ->findOrFail($user_id);

            $the_user_assigned_roles = $the_user->siteRoles()->pluck('role_id');


            $roles = $this->role
                ->all();

            return response()->json([$roles, $the_user_assigned_roles], 200, []);
        } catch (Exception $exception) {
            return JsonResponse::create(['message' => $exception->getMessage()], 410);
        }
    }


    public function getRolesWithoutAdminRole()
    {

        $roles = SiteRole::where('key', "!=", 'admin')->get();
        return $roles;
    }


    public function rearrangeSidebar($role_id)
    {
        $get_all_users_with_role = SiteRole::findOrFail($role_id)->users()->get() ?? null;

        foreach ($get_all_users_with_role as $k => $value) {
            $user_id = $value->id;
            if (file_exists(base_path("resources/views/default/admin/layouts/sidebars/" . Session::get('ALIAS1') . '/' . $user_id . '.blade.php'))) {
                unlink(base_path("resources/views/default/admin/layouts/sidebars/" . Session::get('ALIAS1') . '/' . $user_id . '.blade.php'));
            }
        }
    }


}
