<?php

namespace App\Http\Controllers\Site\Payment;

use App\Models\Academic\AcademicMonthName;
use App\Models\Edu\SitePayment\EduInvoicePayment;
use App\Models\Edu\SitePayment\SiteInvoice;
use App\Models\Edu\SitePayment\SiteInvoiceDetail;
use App\Models\Site\SiteInfo;
use App\Models\Site\SiteMembership;
use App\Traits\Invoice\InvoiceHistoryTrait;
use App\Traits\SmsApiTrait;
use Auth;
use Carbon\Carbon;
use DB;
use Error;
use Exception;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use App\Http\Controllers\Controller;
use Illuminate\Validation\ValidationException;
use PDF;
use Session;

class SiteInvoiceController extends Controller
{
    use InvoiceHistoryTrait, SmsApiTrait;
    private $sitePayment;
    private $siteMembership;
    private $siteInfo;
    private $invoice;
    private $invoiceDetail;

    public function __construct(SiteInvoice $invoice, SiteInvoiceDetail $invoiceDetail, EduInvoicePayment $sitePayment, SiteInfo $siteInfo, SiteMembership $siteMembership)
    {
        $this->middleware('auth');
        $this->middleware('sitepagechecker');
        $this->siteInfo = $siteInfo;
        $this->siteMembership = $siteMembership;
        $this->sitePayment = $sitePayment;
        $this->invoice = $invoice;
        $this->invoiceDetail = $invoiceDetail;

    }


    /* Invoice status
 draft==when first created
 viewed=client viewed it
 approved=admin approve it
 Partial=payment partial
 Paid=full paid*/


    /**
     * @return \Illuminate\Contracts\View\Factory|\Illuminate\View\View
     */
    public function index()
    {

        /*
                $rr=     [
                    'sms_txt' => 'Login Info[nl]Hasan Uddin[nl]http://ctgcs.eduworld.test[nl]Username: 2000900001[nl]Password: 8JsP7n[nl]Regards: Chattogram Collegiate School',
                    'sms_bearer' => 'site',
                    'sms_bearer_id' => 9,
                    'sms_company' => 'zamanit',
                    'per_sms_cost' => '0.35',
                    'sms_count' => 1.0,
                    'total_sms_charges' => 0.35,
                    'sms_type' => 'text',
                    'sms_masking' => NULL,
                ];

                $tt=    $this->sendSmsDynamicApi(['01675020403'],$rr);


              //  dd($tt);*/

        $viewType = 'Site Invoice';
        return view('default.admin.layouts.master', compact('viewType'));
        return view('default.admin.edu.site-payment.index', compact('viewType'));
    }


    public function create(Request $request)
    {
        DB::beginTransaction();

        try {


            $this->validate($request, [
                'invoice_date' => 'required',
                'due_date' => 'required',
                'sub_total' => 'required',
                'site_id' => 'required|numeric|min:1',
                'grand_total' => 'required|numeric|min:1'
            ]);


            $user = request()->user()->id;
            $site_id = $request->site_id;
            $items = $request->details;

            $previous_due_id = $request->previous_due_id;


            if (!$site_id) {
                throw new Error('Please Choose Site!');
            }
            if (count($items) < 1) {
                throw new Error('Please Choose Item!');
            }


            $invoice_number = $this->invoiceNumberGeneration();

            $check_invoice = $this->invoice
                ->where("invoice_no", $invoice_number)
                ->first();

            if ($check_invoice) {
                $invoice_number = $this->invoiceNumberGeneration();
            }


            if (isset($this->invoice->ownFields)) {
                foreach ($this->invoice->ownFields as $ownField) {
                    if ($request->{$ownField}) {
                        $this->invoice->{$ownField} = $request->{$ownField};
                    }
                }
                $this->invoice->invoice_no = $invoice_number;
                $this->invoice->status_code = 'approved';
                $this->invoice->created_by = $user;
                $this->invoice->save();


                foreach ($items as $k => $item) {

                    $this->invoiceDetail = new SiteInvoiceDetail();

                    if (count($this->invoiceDetail->ownFields) > 0) {
                        foreach ($this->invoiceDetail->ownFields as $ownField) {
                            if ($item[$ownField]) {
                                $this->invoiceDetail->{$ownField} = $item[$ownField];
                            }
                        }
                    }
                    $this->invoiceDetail->invoice_id = $this->invoice->id;
                    $this->invoiceDetail->item_details_id = $item['item_details_id'];
                    $this->invoiceDetail->save();
                }
            }

            $this->createHistory($this->invoice->id, 'draft', 'New Invoice:' . $this->invoice->invoice_no . ' created Successfully');

            //check previous due_id
            if ($previous_due_id) {
                $pre_invoice = $this->invoice
                    ->where("id", $previous_due_id)
                    ->first();

                $pre_invoice->carried_invoice_id = $this->invoice->id;
                $pre_invoice->status_code = 'carried';
                $pre_invoice->save();

                $this->createHistory($pre_invoice->id, 'carried', 'The Invoice:' . $pre_invoice->invoice_no . ' Carried to ' . $this->invoice->invoice_no . ' Successfully');
            }

            DB::commit();
            return response()->json(['message' => $this->invoice->invoice_no . ' Successfully Created!'], 200, []);

        } catch (ValidationException $exception) {
            return JsonResponse::create(['message' => $exception->getMessage(), 'errors' => $exception->validator->getMessageBag()->toArray()], 422);
        } catch (Exception $exception) {
            DB::rollback();
            throw new Error($exception->getMessage());
        }
    }

    public function update(Request $request)
    {
        DB::beginTransaction();
        try {


            $this->validate($request, [
                'invoice_date' => 'required',
                'due_date' => 'required',
                'sub_total' => 'required',
                'site_id' => 'required|numeric|min:1',
                'grand_total' => 'required|numeric|min:1'
            ]);


            $user = request()->user()->id;
            $site_id = $request->site_id;
            $items = $request->details;


            if (!$site_id) {
                throw new Error('Please Choose Site!');
            }
            if (count($items) < 1) {
                throw new Error('Please Choose Item!');
            }

            $this->invoice = $this->invoice
                ->where("id", $request->id)
                ->first();

            if (count($this->invoice->payments)) {
                throw new Error('The invoice has payment ' . $this->invoice->payments->sum('paid_amount') . ' Tk.');
            }
            if ($this->invoice->status_code == 'Paid') {
                throw new Error('The invoice is paid already!!');
            }


            if (isset($this->invoice->ownFields)) {
                foreach ($this->invoice->ownFields as $ownField) {
                    if ($request->{$ownField}) {
                        $this->invoice->{$ownField} = $request->{$ownField};
                    }
                };
                //$this->invoice->created_by = $user;
                $this->invoice->save();
                $this->invoice->details()->delete();


                foreach ($items as $k => $item) {

                    $this->invoiceDetail = new SiteInvoiceDetail();

                    if (count($this->invoiceDetail->ownFields) > 0) {
                        foreach ($this->invoiceDetail->ownFields as $ownField) {
                            if ($item[$ownField]) {
                                $this->invoiceDetail->{$ownField} = $item[$ownField];
                            }
                        }
                    }
                    $this->invoiceDetail->invoice_id = $this->invoice->id;
                    $this->invoiceDetail->item_details_id = $item['item_details_id'];
                    $this->invoiceDetail->save();
                }
            }

            $this->createHistory($this->invoice->id, 'draft', ' Invoice:' . $this->invoice->invoice_no . ' Updated Successfully. grand total: '.$this->invoice->grand_total);


            DB::commit();
            return response()->json(['message' => $this->invoice->invoice_no . ' Successfully Updated!'], 200);

        } catch (ValidationException $exception) {
            return JsonResponse::create(['message' => $exception->getMessage(), 'errors' => $exception->validator->getMessageBag()->toArray()], 422);
        } catch (Exception $exception) {
            DB::rollback();
            throw new Error($exception->getMessage());
        }
    }


    public function deleteSiteInvoice($id)
    {
        try {

            $the_invoice = $this->invoice
                ->findOrFail($id);

            if (count($the_invoice->payments) > 0) {
                return response()->json(['message' => 'The invoice can"t Be Deleted'], 410, []);
            }

            $the_invoice->delete();

            $this->createHistory($the_invoice->id, 'carried', 'The Invoice:' . $the_invoice->invoice_no . ' Successfully Deleted By: ' . Auth::user()->id);


            return response()->json(['message' => 'The invoice Deleted Successfully'], 200, []);


        } catch (Exception $exception) {
            throw new Error($exception->getMessage());
        }

    }


    public function getTheInvoice($id)
    {
        try {

            $the_invoice = $this->invoice
                ->with('details')
                ->findOrFail($id);
            return response()->json($the_invoice, 200);


        } catch (Exception $exception) {
            throw new Error($exception->getMessage());
        }

    }


    public function getInvoiceBySearch(Request $request)
    {
        //  \Log::info();


        $site_id = $request->site_id;
        $paginate = $request->paginate ?? 20;
        $invoice_no = $request->invoice_no;

        if(isset($request->date_range) && is_array($request->date_range)) {
            $start_date = isset($request->date_range['start']) ? $request->date_range['start'] : null;
            $end_date = isset($request->date_range['end']) ? $request->date_range['end'] : null;
        } else {
            // Handle the case where $request->date_range is null or not an array
            $start_date = null;
            $end_date = null;
        }

        if (Session::get('SITE_ID')) {
            $site_id = Session::get('SITE_ID');
        }


        if ($request->date_range) {
            if (!$start_date || !$end_date) {
                throw new Error('Please Select Both Date');
            }
        }

        $list = $this->invoice
            ->when($site_id, function ($query) use ($site_id) {
                return $query->where('site_id', $site_id);
            })->when(isset($start_date) && isset($end_date), function ($query) use ($start_date, $end_date) {
                return $query->whereBetween('invoice_date', [$start_date, $end_date]);
            })->when($invoice_no, function ($query) use ($invoice_no) {
                return $query->where('invoice_no', 'like', '%' . $invoice_no . '%');
            })
            ->orderBy('id', 'DESC')
            ->paginate($paginate);

        return response()->json([$list], 200);

    }

    public function getInvoiceList(Request $request)
    {

        $site_id    = $request->site_id;
        $paginate   = $request->paginate ?? 20;
        $invoice_no = $request->invoice_no;


        // Decode the JSON string into an associative array
        $date_range_array = json_decode($request->input('date_range'), true);
        if ($date_range_array !== null) {
            $start_date = isset($date_range_array['start']) ? $date_range_array['start'] : null;
            $end_date = isset($date_range_array['end']) ? $date_range_array['end'] : null;
        } else {
            $start_date = null;
            $end_date = null;
        }

        if ($date_range_array) {
            if (!$start_date || !$end_date) {
                throw new Error('Please Select Both Date');
            }
        }

        $list = $this->invoice
            ->when(isset($site_id), function ($query) use ($site_id) {
                return $query->where('site_id', $site_id);
            })
            ->when(isset($start_date) && isset($end_date), function ($query) use ($start_date, $end_date) {
                return $query->whereBetween('invoice_date', [$start_date, $end_date]);
            })
            ->when(isset($invoice_no), function ($query) use ($invoice_no) {
                return $query->where('invoice_no', 'like', '%' . $invoice_no . '%');
            })
            ->orderBy('id', 'DESC')
            ->paginate($paginate);


        return response()->json([$list], 200);

    }


    public function getInvoiceListBySite(Request $request)
    {

        $invoice_no = $request->invoice_no;
        $paginate = $request->paginate ?? 20;

        if(isset($request->date_range) && is_array($request->date_range)) {
            $start_date = isset($request->date_range['start']) ? $request->date_range['start'] : null;
            $end_date = isset($request->date_range['end']) ? $request->date_range['end'] : null;
        } else {
            // Handle the case where $request->date_range is null or not an array
            $start_date = null;
            $end_date = null;
        }

        if ($request->date_range) {
            if (!$start_date || !$end_date) {
                throw new Error('Please Select Both Date');
            }
        }

        $list = $this->invoice
            ->when(isset($start_date) && isset($end_date), function ($query) use ($start_date, $end_date) {
                return $query->whereBetween('invoice_date', [$start_date, $end_date]);
            })->when($invoice_no, function ($query) use ($invoice_no) {
                return $query->where('invoice_no', 'like', '%' . $invoice_no . '%');
            })
            ->where('site_id', Session::get('SITE_ID'))
            ->orderBy('id', 'DESC')
            ->paginate($paginate);


        return response()->json([$list], 200);

    }

    /**
     * @return \Illuminate\Contracts\View\Factory|\Illuminate\View\View
     */
    public function getDataForNewInvoiceBySite($type, $site_id)
    {
        try {


            $the_site = $this->siteInfo->where('id', $site_id)->first();
            $site_membership = $the_site->siteMembership;


            if (!$site_membership) {
                throw new Error('No Site MemberShip Found For The Site!');
            }
            $last_invoice = null;
            if ($type == 1) {//with due
                $last_invoice = $this->invoice
                    ->where('site_id', $site_id)
                    ->whereIn('status_code', ['partial', 'approved'])
                    ->orderBy('id', 'DESC')
                    ->first();
            }


            $months = AcademicMonthName::all();

            //dd($last_invoice);


            $data = [];
            //$data['invoice_no'] = $invoice_no;
            $data['previous_due'] = $last_invoice->amount_due ?? 0;
            $data['previous_due_id'] = $last_invoice->id ?? 0;
            $data['site_membership'] = $site_membership;
            $data['months'] = $months;

            return response()->json($data, 200, []);
        } catch (Exception $exception) {
            return response()->json($exception->getMessage(), 411, []);

        }


    }

    public function getInvoiceDetailsForPayment($id)
    {
        $list = $this->invoice
            ->with(['details.item', 'details.month', 'siteInfo'])
            ->where('id', $id)
            ->first();
        return response()->json($list, 200);

    }


    public function getInvoiceHistories($id)
    {
        try {
            $list = $this->invoice
                ->with(['histories', 'siteInfo'])
                ->where('id', $id)
                ->first();
            return response()->json($list, 200, []);
        } catch (Exception $exception) {
            return response()->json($exception->getMessage(), 411, []);

        }

    }

    private function invoiceNumberGeneration($invoice_number = null)
    {

        $new_invoice = null;
        $initial_part = 'EINV-';
        $part_length = strlen($initial_part);


        if ($invoice_number) {

            $current_number = (int)substr($invoice_number, $part_length);
            $next_number = $current_number + 1;
            $new_invoice = $initial_part . sprintf("%06d", $next_number);

            // dd($invoice_number);

        } else {

            $invoice_existed = $this->invoice
                ->whereRaw("invoice_no REGEXP '^{$initial_part}([0-9]*)?$'")
                ->orderBy('id', 'DESC')
                ->first();

            //dd($invoice_existed);

            if ($invoice_existed) {
                $current_number = (int)substr($invoice_existed->invoice_no, $part_length);
                $next_number = $current_number + 1;
                $new_invoice = $initial_part . sprintf("%06d", $next_number);

            } else {
                $new_invoice = $initial_part . sprintf("%06d", 1);
            }
        }

        $check_invoice = $this->invoice
            ->where("invoice_no", $new_invoice)
            ->first();

        if ($check_invoice) {
            $invoice_no = ($check_invoice->invoice_no) ?? ($initial_part . sprintf("%06d", 1));
            return $this->invoiceNumberGeneration($invoice_no);

        } else {
            return $new_invoice;
        }


    }


    /**
     * @param $id
     */
    public function getInvoicePdfReport($id)
    {
        try {


            if (Session::get('USER_TYPE') == 'eduworld') {

                $the_invoice = $this->invoice
                    ->findOrFail($id);

            } else {


                $the_invoice = $this->invoice
                    ->where('id', $id)
                    ->where('site_id', Session::get('SITE_ID'))
                    ->first();

            }


//        dd($the_invoice->payments->last()->payment_date ??  '');
            $report_tittle = 'Invoice Report-' . $the_invoice->invoice_no;


            // return view('default.admin.edu.site-payment.payment-report', compact('report_tittle', 'the_invoice'));
            $pdf = PDF::loadView('default.admin.edu.site-payment.payment-report', compact('report_tittle', 'the_invoice'));
            return $pdf->stream(time() . '-invoice-no-' . $id . '.pdf');
        } catch (Exception $exception) {
            return $exception->getMessage();
        }
    }
}
