<?php

namespace App\Http\Controllers\Site\Library;

use App\Http\Controllers\Controller;
use App\Models\Author\Author;
use App\Models\Book\Book;
use App\Models\Book\BookBorrowing;
use App\Models\Book\BookCategory;
use App\Models\Book\BookReck;

use App\Models\Book\BookReckAssign;

use App\Models\Book\LibraryBookHistory;
use App\Models\Book\LibraryFineConfiguration;
use App\Models\Book\LibraryMember;

use App\Models\SiteGroup\Site\Employee\Employee;
use App\Models\SiteGroup\Site\Student\Student;

use App\Models\Student\StudentHistory;
use App\Models\User\EmployeeHistory;
use Carbon\Carbon;
use Illuminate\Http\Request;

use Mpdf\Tag\Em;
use Illuminate\Support\Str;

use Illuminate\Support\Facades\DB;
use PDF;

class SiteLibraryController extends Controller
{
    public function __construct()
    {
        $this->middleware('auth');
        // $this->middleware('sitepagechecker');
    }

    public function index()
    {
        $viewType = 'page';
        return view('default.admin.layouts.master', compact('viewType'));
    }
    public function store(Book $book, Request $request)
    {
        DB::transaction(function () use ($request, $book) {

            $request->validate([
                'quantity'          => 'required|integer|min:1',
                'name'              => 'required',
                'author'            => 'required',
                'category'          => 'required',
                'book_reck_id'      => 'nullable|integer|exists:book_recks,id',
                'quantity'          => 'required|integer|min:0',
            ]);


            $book->fill($request->only($book->getFillable()));


            $book->book_name            = $request->get('name');
            $book->author_id            = $request->get('author');
            $book->category_id          = $request->get('category');
            $book->isbn                 = $request->get('isbn');
            $book->damage_info          = $request->get('damageInfo');
            $book->sale_status          = $request->get('saleStatus');
            $book->book_reck_id         = $request->get('book_reck_id');
            $book->book_reck_shelf_name = $request->get('book_reck_shelf_name');
            $book->available_quantity   = $request->get('availableQuantity');
            $book->location             = $request->get('location');

        

            if ($book->save()) {

                $quantity = $request->get('quantity');

                $rack = BookReck::find($request->get('book_reck_id'));

                if ($rack) {
                    if (($rack->currentBooks + $quantity) > $rack->capacity) {
                        throw new \Exception('Rack capacity exceeded. Cannot add more books to this rack.');
                    }

                    if ($quantity) {
                        $rack->currentBooks += $quantity;
                        $rack->save();
                    }
                }


                if ($quantity) {



                    $categoryName = BookCategory::find($request->get('category'))->name;

                    $prefix = strtoupper(substr($categoryName, 0, 3));


                    $baseCode =  $prefix . $book->id;

                    for ($i = 0; $i < $quantity; $i++) {
                        $uniqueCode = $this->generateUniqueBookCode($baseCode, $i + 1);

                        LibraryBookHistory::create([
                            'book_code' => $uniqueCode,
                            'book_id'   => $book->id,
                            'status'   => 'Available',
                            'condition'   => 'New',

                        ]);
                    }
                }
            }

            return response()->json(['success' => true, 'message' => 'Book saved successfully.']);
        });
    }
    public function update(Request $request)
    {
        DB::beginTransaction(function () use ($request) {
            try {
                $book =  Book::find($request->get('id'));
                $temp_quantity = $book->quantity;

                if ($request->get('quantity') < $temp_quantity) {
                    throw new \Exception('You can not reduce the quantity. Please contact admin.');
                }

                $request->validate([
                    'quantity'     => 'required|integer|min:1',
                    'name'         => 'required',
                    'author'       => 'required',
                    'category'     => 'required',
                    'book_reck_id' => 'nullable|integer|exists:book_recks,id',
                    'quantity'     => 'required|integer|min:0',
                    'notes'        => 'required',
                ]);


                $book->fill($request->only($book->getFillable()));

                $book->book_name            = $request->get('name');
                $book->author_id            = $request->get('author');
                $book->category_id          = $request->get('category');
                $book->isbn                 = $request->get('isbn');
                $book->damage_info          = $request->get('damageInfo');
                $book->sale_status          = $request->get('saleStatus');
                $book->book_reck_id         = $request->get('book_reck_id');
                $book->book_reck_shelf_name = $request->get('book_reck_shelf_name');
                $book->available_quantity   = $request->get('availableQuantity');
                $book->location             = $request->get('location');

                if ($book->save()) {

                    if ($request->quantity) {
                        if ($request->get('quantity') > $temp_quantity) {
                            $quantity =  $request->get('quantity') - $temp_quantity;
                            $baseCode = "BK" . $book->id;
                            for ($i = 0; $i < $quantity; $i++) {
                                $uniqueCode = $this->generateUniqueBookCode($baseCode, $i + 1);

                                LibraryBookHistory::create([
                                    'book_code' => $uniqueCode,
                                    'book_id'   => $book->id,
                                    'status'   => 'Available',
                                    'condition'   => 'New',
                                ]);
                            }
                        }
                    }
                }

                return response()->json(['success' => true, 'message' => 'Book saved successfully.']);
            } catch (\Exception $e) {
                DB::rollBack();
                throw $e; // Rethrow the exception after rollback
            }
        });
    }
    public function getAvailableBooks()
    {
        return Book::with(['bookAuthor', 'bookHistory'])->get()->transform(function ($book) {
            return [
                'id'                 => $book->id,
                'title'              => $book->title,
                'libraryBookHistory' => $book->bookHistory ?? [],
                'name'               => $book->book_name,
                'author'             => $book->bookAuthor->name ?? 'Unknown',
                'quantity'           => $book->quantity ?? '',
                'available_quantity' => $book->available_quantity ?? 0,
            ];
        });
    }
    public function GetBooksAuthors(Request $request)
    {
        $authors = Author::all();
        return response()->json($authors);
    }
    public function GetBooksCategory(Request $request)
    {
        $bookCategories = BookCategory::query()
            ->when($request->search, function ($q) use ($request) {
                $search = $request->search;
                $q->where('name', 'like', "%{$search}%");
            })->get();



        $bookCategories->transform(function ($category) {
            return [
                'id'          => $category->id,
                'name'        => $category->name,
                'description' => $category->description ?? '',
                'status'      => $category->status ?? 'active',
            ];
        });

        return response()->json($bookCategories);
    }
    public function storeCategory(Request $request)
    {
        $Category = new BookCategory();
        $request->validate([
            'name'              => 'required',
        ]);
        $Category->fill($request->only($Category->getFillable()));
        $Category->save();
        return response()->json(['success' => true, 'message' => 'Book category saved successfully.']);
    }
    public function UpdateBooksCategory($id, Request $request)
    {
        $request->validate([
            'name' => 'required',
        ]);

        $Category = BookCategory::find($id);

        if (!$Category) {
            return response()->json(['success' => false, 'message' => 'Category not found.'], 404);
        }

        $Category->update($request->only(['name', 'description']));

        return response()->json(['success' => true, 'message' => 'Book category updated successfully.']);
    }
    public function updateBookStatusCondition(Request $request)
    {


        $bookHistory = LibraryBookHistory::find($request->get('id'));
        if (!$bookHistory) {
            return response()->json(['success' => false, 'message' => 'Book history not found.'], 404);
        }

        $bookHistory->status    = $request->get('status');
        $bookHistory->condition = $request->get('condition');
        $bookHistory->save();

        return response()->json(['success' => true, 'message' => 'Book status and condition updated successfully.']);
    }
    public function BookStatisticsCount()
    {
        $totalBooks = Book::count();
        $availableBooks = Book::sum('available_quantity');
        $damagedBooks = LibraryBookHistory::where('condition', 'Damaged')->count();
        $checkedOutBooks = BookBorrowing::where('returned_status', '!=', 'returned')->count();
        $categories = BookCategory::count();

        return response()->json([
            'totalBooks' => $totalBooks,
            'availableBooks' => $availableBooks,
            'damagedBooks' => $damagedBooks,
            'checkedOutBooks' => $checkedOutBooks,
            'categories' => $categories,
        ]);
    }


    function generateUniqueBookCode($baseCode, $index = 1)
    {
        do {
            // Generate a 4-digit random number
            $randomNumber = mt_rand(100000, 999999);
            $code = $baseCode . '-' . $index . '-' . $randomNumber;
        } while (LibraryBookHistory::where('book_code', $code)->exists());

        return $code;
    }

    public function GetWithoutAssignBooks(Request $request)
    {
        return $query = Book::whereNull('book_reck_id')->get();
    }
    public function GetBooks(Request $request)
    {
        $query = Book::query()
            ->with('bookAuthor', 'bookCategory', 'bookHistory')
            ->when($request->category, fn($q) => $q->where('category_id', $request->category))
            ->when($request->status, fn($q) => $q->where('status', $request->status))
            ->when($request->condition, fn($q) => $q->where('condition', $request->condition))
            ->when($request->sale, fn($q) => $q->where('sale_status', $request->sale))
            ->when($request->search, function ($q) use ($request) {
                $search = $request->search;
                $q->where(function ($q2) use ($search) {
                    $q2->where('book_name', 'like', "%{$search}%")
                        ->orWhere('title', 'like', "%{$search}%")
                        ->orWhere('subject', 'like', "%{$search}%")
                        ->orWhere('isbn', 'like', "%{$search}%")
                        ->orWhere('publisher', 'like', "%{$search}%");
                });
            });

        $libraryBooks = $query->paginate(15);

        $libraryBooks->transform(function ($book) {

            $BookReck = BookReck::find($book->book_reck_id);
            $book_location = $BookReck ? "<b>" . $BookReck->name . "</b>" . "<br>" : 'Unassigned';
            $book_location .= $book->book_reck_shelf_name ? 'Shelf: ' . $book->book_reck_shelf_name : '';

            return [
                'id'                => $book->id,
                'name'              => $book->book_name,
                'libraryBookHistory' => $book->bookHistory ?? 'N/A',
                'title'             => $book->title ?? 'N/A',
                'author'            => $book->bookAuthor->name ?? 'Unknown',
                'author_id'            => $book->bookAuthor->id ?? '',
                'subject'           => $book->subject ?? 'N/A',
                'category'          => $book->bookCategory->name ?? 'Uncategorized',
                'category_id'          => $book->bookCategory->id ?? '',
                'subCategory'       => $book->subCategory ?? 'N/A',
                'isbn'              => $book->isbn ?? 'N/A',
                'pages'             => $book->pages ?? 0,
                'year'              => $book->year ?? 'N/A',
                'book_reck_id'      => $book->book_reck_id ?? '',
                'publisher'         => $book->publisher ?? 'N/A',
                'quantity'          => $book->quantity ?? 0,
                'availableQuantity' => $book->available_quantity ?? 0,
                'location'          => $book_location ?? 'Unknown',
                'status'            => $book->status ?? 'Available',
                'condition'         => $book->condition ?? 'Good',
                'saleStatus'        => $book->sale_status ?? 'Not for Sale',
                'price'             => $book->price ?? null,
                'notes'             => $book->notes ?? '',
                'damageInfo'        => $book->damage_info ?? '',
            ];
        });

        return response()->json($libraryBooks);
    }
    public function delete(Request $request)
    {
        $id = $request->id;
        $Book = Book::find($id);

        if (!$Book) {
            return response()->json(['success' => false, 'message' => 'Book not found.']);
        } else {
            $Book->delete();
            return 'Book deleted successfully.';
        }
    }


    // author

    public function showAuthors()
    {
        $authors = Author::all();
        return response()->json($authors);
    }

    public function storeAuthors(Request $request)
    {
        $validated = $request->validate([
            'firstName' => 'required|string|max:100',
            'gender'    => 'required|in:male,female,other',
            'birthDate' => 'required',
        ]);

        $author = Author::create($request->all());
        return response()->json($author, 201);
    }

    public function updateAuthors(Request $request, $id)
    {
        $author = Author::findOrFail($id);

        $validated = $request->validate([
            'firstName' => 'required|string|max:100',
            'gender'    => 'required|in:male,female,other',
            'birthDate' => 'required',
        ]);

        $author->update($request->all());
        return response()->json($author);
    }

    public function destroyAuthors($id)
    {
        $author = Author::findOrFail($id);
        $author->delete();
        return response()->json(['message' => 'Author deleted successfully.']);
    }

    public function GetBookRacks()
    {
        return BookReck::all();
    }
    public function showRacks()
    {
        return  response()->json(BookReck::all());
    }

    public function storeRack(Request $request)
    {


        $validated = $request->validate([
            'code'        => 'required|unique:book_recks,code',
            'name'        => 'required|string',
            'location'    => 'nullable|string',
            'capacity'    => 'required|integer|min:1',
            'category'    => 'required|string',
            'description' => 'nullable|string',
            'description' => 'nullable|string',
        ]);

        $validated['currentBooks'] = 0;
        $validated['site_id'] = session()->get("SITE_ID");
        $validated['rack_shelf'] = $request->rack_shelf ?? null;


        return BookReck::create($validated);
    }

    public function updateRack(Request $request, $id)
    {
        $rack = BookReck::findOrFail($id);

        $validated = $request->validate([
            'code'        => 'required|unique:book_recks,code,' . $rack->id,
            'name'        => 'nullable|string',
            'location'    => 'nullable|string',
            'capacity'    => 'required|integer|min:1',
            'category'    => 'nullable|string',
            'description' => 'nullable|string',
        ]);
        $validated['rack_shelf'] = $request->rack_shelf ?? null;

        $rack->update($validated);
        return $rack;
    }

    public function destroyRack($id)
    {
        $rack = BookReck::findOrFail($id);
        $rack->delete();

        return response()->json(['message' => 'Rack deleted successfully']);
    }

    public function assignBooks(Request $request)
    {
        $validated = $request->validate([
            'rack_id'    => 'required|exists:book_recks,id',
            'book_ids'   => 'required|array',
            'book_ids.*' => 'exists:books,id', // Assuming book model exists
        ]);

        foreach ($validated['book_ids'] as $bookId) {
            Book::find($bookId)->update([
                'book_reck_id' => $validated['rack_id'],
            ]);
        }

        $rack = BookReck::find($validated['rack_id']);
        $rack->currentBooks += count($validated['book_ids']);
        $rack->save();

        return response()->json(['message' => 'Books assigned successfully']);
    }

    public function GetStudentStaffTeacherPublic(Request $request)
    {
        $memberType = $request->type  ?? 'student';
        if ($memberType == 'student') {



            if (!$request->academic_class_id || !$request->academic_year_id || !$request->academic_version_id) {
                return response()->json(['error' => 'Academic class, year, or version is required'], 400);
            }
            $user_ids = LibraryMember::where('type', 'student')->pluck('user_id')->filter(); // avoid nulls

            $students = StudentHistory::query()
                ->with(['stClass', 'classGroup'])
                ->when($request->academic_class_id, fn($q) => $q->where('academic_class_id', $request->academic_class_id))
                ->when($request->academic_year_id, fn($q) => $q->where('academic_year_id', $request->academic_year_id))
                ->when($request->academic_version_id, fn($q) => $q->where('academic_version_id', $request->academic_version_id))
                ->where('status', 'active')
                ->when($user_ids->isNotEmpty(), function ($q) use ($user_ids) {
                    $q->whereNotIn('user_id', $user_ids);
                })
                ->orderBy('student_roll_number', 'asc')
                ->get()
                ->transform(function ($student) {
                    return [
                        'id'         => $student->id,
                        'username'   => $student->username,
                        'user_id'    => $student->user_id,
                        'name'       => $student->full_name . ' (' . ($student->stClass->class_name ?? 'N/A') . ') - ' . ($student->classGroup->group_name ?? 'N/A'),
                        'class'      => $student->stClass->class_name ?? 'N/A',
                        'stClass'    => $student->stClass ?? 'N/A',
                        'classGroup' => $student->classGroup ?? 'N/A',
                        'roll_no'    => $student->student_full_roll_number ?? 'N/A',
                    ];
                });

            return response()->json($students);
        } else if ($memberType == 'staff') {
            $user_ids = LibraryMember::where('type', 'staff')->pluck('user_id')->filter();
            $staffs = EmployeeHistory::query()
                ->where('is_teacher', false)
                ->with('employee')
                ->where('status', 'active')
                ->when($user_ids->isNotEmpty(), function ($q) use ($user_ids) {
                    $q->whereNotIn('user_id', $user_ids);
                })
                ->get()->transform(function ($staff) {
                    return [
                        'id'          => $staff->id,
                        'user_id'     => $staff->user_id,
                        'username'    => $staff->employee->username ?? 'N/A',
                        'name'        => $staff->full_name . ' (' . ($staff->designation_name ?? 'N/A') . ')',
                        'designation' => $staff->designation_name ?? 'N/A',
                    ];
                });
            return response()->json($staffs);
        } else if ($memberType == 'teacher') {
            $user_ids = LibraryMember::where('type', 'teacher')->pluck('user_id')->filter();
            $teachers = EmployeeHistory::query()
                ->with('employee')
                ->where('is_teacher', true)
                ->where('status', 'active')
                ->when($user_ids->isNotEmpty(), function ($q) use ($user_ids) {
                    $q->whereNotIn('user_id', $user_ids);
                })

                ->get()->transform(function ($teacher) {
                    return [
                        'id'          => $teacher->id,
                        'user_id'     => $teacher->user_id,
                        'username'    => $teacher->employee->username ?? 'N/A',
                        'name'        => $teacher->full_name . ' (' . ($teacher->designation_name ?? 'N/A') . ')',
                        'designation' => $teacher->designation_name ?? 'N/A',

                    ];
                });
            return response()->json($teachers);
        }
    }
    public function storeMember(Request $request)
    {
        $member =  $request->member;

        $memberType = $request->type;

        if ($memberType == 'student') {
            $request->validate([
                'member' => 'required',
            ]);
            LibraryMember::create([
                'type' => $memberType,
                'status' => $request->status,
                'user_id' => $member['user_id'],
                'member_id' => $member['username'],
            ]);
        } else if ($memberType == 'staff') {
            $request->validate([
                'member' => 'required',
            ]);
            LibraryMember::create([
                'type' => $memberType,
                'status' => $request->status,
                'user_id' => $member['user_id'],
                'member_id' => $member['username'],
            ]);
        } else if ($memberType == 'teacher') {
            $request->validate([
                'member' => 'required',
            ]);
            LibraryMember::create([
                'type' => $memberType,
                'status' => $request->status,
                'user_id' => $member['user_id'],
                'member_id' => $member['username'],
            ]);
        } else if ($memberType == 'public') {
            $request->validate([
                'name' => 'required|string|max:255',
                'email' => 'email|unique:library_members,email',
                'phone' => [
                    'required',
                    'regex:/^(?:\+88|88)?01[3-9]\d{8}$/',
                ],
                'address' => 'required',
            ]);

            LibraryMember::create([
                'type' => $memberType,
                'status' => $request->status,
                'member_id' => 'P' . date('Y') . $this->generateUniqueId('library_members', 'member_id', 6),
                'name' => $request->name,
                'email' => $request->email,
                'phone' => $request->phone,
                'address' => $request->address,
            ]);
        }

        return 'done';
    }
    function generateUniqueId($table, $column = 'unique_id', $length = 6)
    {
        do {

            $timePart = date('is');
            $timePart = substr($timePart, -2);

            // Generate random digits
            $randomPart = '';
            for ($i = 0; $i < $length; $i++) {
                $randomPart .= mt_rand(0, 9);
            }


            $uniqueId = $timePart . $randomPart;

            $exists = DB::table($table)->where($column, $uniqueId)->exists();
        } while ($exists);

        return $uniqueId;
    }

    public function getLibraryMembersPdf(Request $requst)
    {
        $LibraryMember =  LibraryMember::find($requst->id);

        $report_tittle = "";
        $page_tittle  = "";
        $report_tittle2   = "";


        ini_set('memory_limit', '2048M');
        $pdf = PDF::loadView(
            'default.site.library.library-member-pdf',
            [
                'member'         => $LibraryMember,
                'report_tittle'  => $report_tittle,
                'page_tittle'    => $page_tittle,
                'report_tittle2' => $report_tittle2,
            ]
        )->setPaper('a4', 'portrait')->setWarnings(false);

        $random = rand(0, 100000);
        $time = $random . '-' . Carbon::now()->unix();
        $name = $time . '.pdf';

        return $pdf->stream($name);
    }
    public function getMember()
    {
        $libraryMembers = LibraryMember::paginate(15);

        $libraryMembers->getCollection()->transform(function ($member) {

            $BookBorrowing = BookBorrowing::whereNull('returned_date')
                ->where('member_id', $member->id);

            $total_borrowed_books_count = $BookBorrowing->count();
            $total_borrowed_books = $BookBorrowing->count();


            return [
                'id'         => $member->id,
                'type'       => $member->type,
                'status'     => $member->status,
                'full_name'  => $member->full_name,
                'name'       => $member->name,
                'email'      => $member->email,
                'total_borrowed_books_count' => $total_borrowed_books_count,
                'total_borrowed_books' => $total_borrowed_books_count,
                'phone'      => $member->phone,
                'address'    => $member->address,
                'user_id'    => $member->user_id,
                'member_id'  => $member->member_id,
                'created_at' => $member->created_at->format('d M Y'),
            ];
        });

        return response()->json($libraryMembers);
    }
    public function getMemberStatistics()
    {
        $libraryMembers = LibraryMember::get();
        $activelibraryMembers    = $libraryMembers->where('status', 'active')->count();
        $inactivelibraryMembers  = $libraryMembers->where('status', 'inactive')->count();
        $suspendedlibraryMembers = $libraryMembers->where('status', 'suspended')->count();

        $totalCount = $libraryMembers->count();

        $types = ['student', 'staff', 'teacher', 'public'];

        $data = [];

        foreach ($types as $type) {
            $count = $libraryMembers->where('type', $type)->count();
            $data[] = [
                'type'    => $type,
                'count'   => $count,
                'percent' => $totalCount > 0 ? round(($count / $totalCount) * 100) : 0,
            ];
        }

        return response()->json([
            'total' => $totalCount,
            'data'  => $data,
            'activelibraryMembers'    => $activelibraryMembers,
            'inactivelibraryMembers'  => $inactivelibraryMembers,
            'suspendedlibraryMembers' => $suspendedlibraryMembers,
        ]);
    }

    public function getActiveMember()
    {
        $libraryMembers = LibraryMember::where('status', 'active')->get();

        $libraryMembers->transform(function ($member) {
            return [
                'id'         => $member->id,
                'type'       => $member->type,
                'status'     => $member->status,
                'name'       => $member->name,
                'full_name'  => $member->full_name,
                'email'      => $member->email,
                'phone'      => $member->phone,
                'address'    => $member->address,
                'user_id'    => $member->user_id,
                'member_id'  => $member->member_id,
                'created_at' => $member->created_at->format('d M Y'),
            ];
        });

        return response()->json($libraryMembers);
    }

    public function storeBooksBorrowing(Request $request)
    {

        $request->validate([
            'book_id'       => 'required',
            'book_history_id'     => 'required',
            'member_id'     => 'required|exists:library_members,id',
            'borrowed_date' => 'required|date',
            'due_date'      => 'required|date|after_or_equal:borrowed_date',
        ]);

        $memberId = $request->member_id;
        $borrowDate = $request->borrowed_date;
        $dueDate = $request->due_date;
        $bookId = $request->book_id;
        $book_history_id = $request->book_history_id;

        $book = Book::find($bookId);
        if ($book && $book->available_quantity > 0) {
            $borrowing = BookBorrowing::create([
                'member_id'     => $memberId,
                'book_id'       => $bookId,
                'book_history_id' => $book_history_id,
                'borrowed_date' => $borrowDate,
                'due_date'      => $dueDate,
                'status'        => 'active',
            ]);

            // Decrement available quantity
            $book->available_quantity -= 1;
            if ($book->save()) {

                $bookHistory = LibraryBookHistory::find($book_history_id);
                if ($bookHistory) {
                    $bookHistory->status = 'Checked Out';
                    $bookHistory->save();
                }
            }
        }

        return response()->json([
            'success' => true,
            'message' => 'Books borrowed successfully.',
            'data'    => $borrowing,
        ]);
    }
    public function getBorrowingStatistics()
    {
        $totalBorrowings = BookBorrowing::count();

        $currentBorrowings = BookBorrowing::whereNull('returned_date')->count();

        $returnedToday = BookBorrowing::whereDate('returned_date', Carbon::today())->count();

        $overdueBorrowings = BookBorrowing::whereNull('returned_date')
            ->whereDate('due_date', '<', Carbon::today())
            ->count();

        return response()->json([
            'totalBorrowings'   => $totalBorrowings,
            'currentBorrowings' => $currentBorrowings,
            'returnedToday'     => $returnedToday,
            'overdueBorrowings' => $overdueBorrowings,
        ]);
    }

    public function getLibraryBorrowingHistory()
    {
        $today = Carbon::today();

        // Fine configuration
        $fineConfiguration = $this->getLibraryFine();

        // Convert to associative array for easier lookup
        $fineMap = collect($fineConfiguration)->pluck('amount', 'day')->toArray();
        $lastFineAmount = collect($fineMap)->last();

        $BookBorrowing = BookBorrowing::with(['member', 'book.bookAuthor', 'book', 'bookHistory'])
            ->whereNotNull('returned_date')
            ->paginate(15);

        $BookBorrowing->getCollection()->transform(function ($b) use ($fineMap, $lastFineAmount, $today) {

            $dueDate = Carbon::parse($b->due_date);
            $overdueDays = $today->greaterThan($dueDate)
                ? $today->diffInDays($dueDate)
                : 0;

            $totalFine = 0;
            $fineBreakdown = [];

            for ($i = 1; $i <= $overdueDays; $i++) {
                $amount = $fineMap[$i] ?? $lastFineAmount;
                $fineBreakdown[] = [
                    'day' => $i,
                    'amount' => $amount,
                ];
                $totalFine += $amount;
            }
            $book_name = $b->book ? "<b>" . $b->book->book_name . '</b>' . "<br>" : 'N/A';
            $book_name .= $b->book ? 'Author: ' . $b->book->bookAuthor->name . "<br>" : '';
            $book_name .= $b->bookHistory ? 'Book Code: ' . $b->bookHistory->book_code : '';


            return [
                'id'              => $b->id,
                'status'          => $b->status,
                'memberName'      => $b->member->full_name ?? '',
                'memberEmail'     => $b->member->email ?? '',
                'memberPhone'     => $b->member->phone ?? '',
                // 'bookTitle'       => $b->book->title ?? '',
                'fine'            => $b->fine ?? '',
                'bookName'        => $book_name ?? '',
                'bookAuthor'      => $b->book->bookAuthor->name ?? '',
                'borrowed_date'   => $b->borrowed_date,
                'due_date'        => $b->due_date,
                'renewals'        => $b->renewals,
                'returned_date'   => $b->returned_date,
                'returned_status' => $b->returned_status,
                'overdue_days'    => $overdueDays,
                'fine'            => round($totalFine, 2),
                'fine_breakdown'  => $fineBreakdown,
            ];
        });

        return $BookBorrowing;
    }
    public function getOverdueBookBorrowingBookPdfDownload()
    {

        $today = Carbon::today();

        // Fine configuration
        $fineConfiguration = $this->getLibraryFine();

        // Convert to associative array for easier lookup
        $fineMap = collect($fineConfiguration)->pluck('amount', 'day')->toArray();
        $lastFineAmount = collect($fineMap)->last();

        $BookBorrowing = BookBorrowing::with(['member', 'book.bookAuthor'])
            ->whereNull('returned_date')
            ->whereDate('due_date', '<', $today)
            ->get();

        $BookBorrowing->transform(function ($b) use ($fineMap, $lastFineAmount, $today) {
            $dueDate = Carbon::parse($b->due_date);
            $overdueDays = $today->greaterThan($dueDate)
                ? $today->diffInDays($dueDate)
                : 0;
            $isOverdue = $today->greaterThan($dueDate);


            $fineBreakdown = [];

            for ($i = 1; $i <= $overdueDays; $i++) {
                $amount = $fineMap[$i] ?? $lastFineAmount;
                $fineBreakdown[] = [
                    'day'    => $i,
                    'amount' => $amount,
                ];
            }

            return [
                'status'         => $b->status,
                'id'             => $b->id,
                'isOverdue'      => $isOverdue,
                'memberName'     => $b->member->full_name ?? '',
                'memberEmail'    => $b->member->email ?? '',
                'memberPhone'    => $b->member->phone ?? '',
                'bookTitle'      => $b->book->title ?? '',
                'bookName'       => $b->book->book_name ?? '',
                'bookAuthor'     => $b->book->bookAuthor->name ?? '',
                'borrowed_date'  => $b->borrowed_date,
                'due_date'       => $b->due_date,
                'renewals'       => $b->renewals,
                'overdue_days'   => $overdueDays,
                'fine'           => $b->fine ?? 0,
                'fine_breakdown' => $fineBreakdown,
            ];
        });




        $report_tittle = "";
        $page_tittle  = "";
        $report_tittle2   = "";


        ini_set('memory_limit', '2048M');
        $pdf = PDF::loadView(
            'default.site.library.overdue-book-borrowing-book-pdf',
            [
                'BookBorrowing'  => $BookBorrowing,
                'report_tittle'  => $report_tittle,
                'page_tittle'    => $page_tittle,
                'report_tittle2' => $report_tittle2,
            ]
        )->setPaper('a4', 'portrait')->setWarnings(false);

        $random = rand(0, 100000);
        $time = $random . '-' . Carbon::now()->unix();
        $name = $time . '.pdf';

        return $pdf->stream($name);
    }

    public function getOverdueBookBorrowingBook()
    {
        $today = Carbon::today();

        // Fine configuration
        $fineConfiguration = $this->getLibraryFine();

        // Convert to associative array for easier lookup
        $fineMap = collect($fineConfiguration)->pluck('amount', 'day')->toArray();
        $lastFineAmount = collect($fineMap)->last();

        $BookBorrowing = BookBorrowing::with(['member', 'book.bookAuthor', 'book'])
            ->whereNull('returned_date')
            ->whereDate('due_date', '<', $today)
            ->paginate(15);

        $BookBorrowing->getCollection()->transform(function ($b) use ($fineMap, $lastFineAmount, $today) {
            $dueDate = Carbon::parse($b->due_date);
            $overdueDays = $today->greaterThan($dueDate)
                ? $today->diffInDays($dueDate)
                : 0;
            $isOverdue = $today->greaterThan($dueDate);


            $fineBreakdown = [];

            for ($i = 1; $i <= $overdueDays; $i++) {
                $amount = $fineMap[$i] ?? $lastFineAmount;
                $fineBreakdown[] = [
                    'day'    => $i,
                    'amount' => $amount,
                ];
            }

            $book_name = $b->book ? "<b>" . $b->book->book_name . '</b>' . "<br>" : 'N/A';
            $book_name .= $b->book ? 'Author: ' . $b->book->bookAuthor->name . "<br>" : '';
            $book_name .= $b->bookHistory ? 'Book Code: ' . $b->bookHistory->book_code : '';

            return [
                'id'             => $b->id,
                'status'         => $b->status,
                'isOverdue'      => $isOverdue,
                'memberName'     => $b->member->full_name ?? '',
                'memberEmail'    => $b->member->email ?? '',
                'memberPhone'    => $b->member->phone ?? '',
                'bookTitle'      => $b->book->title ?? '',
                'bookName'       => $book_name ?? '',
                'bookAuthor'     => $b->book->bookAuthor->name ?? '',
                'borrowed_date'  => $b->borrowed_date,
                'due_date'       => $b->due_date,
                'renewals'       => $b->renewals,
                'overdue_days'   => $overdueDays,
                'fine'           => $b->fine ?? 0,
                'fine_breakdown' => $fineBreakdown,
            ];
        });

        return $BookBorrowing;
    }
    public function libraryBookBorrowingReturn(Request $request)
    {
        DB::transaction(function () use ($request) {
            $borrowing = BookBorrowing::findOrFail($request->id);

            $today   = Carbon::today();
            $dueDate = Carbon::parse($borrowing->due_date);

            // Update borrowing return info
            $borrowing->update([
                'returned_date'   => $today,
                'returned_status' => $today->greaterThan($dueDate) ? 'returned_late' : 'returned_on_time',
            ]);

            // Update book stock
            Book::where('id', $borrowing->book_id)->increment('available_quantity');

            // Update book history
            LibraryBookHistory::where('id', $borrowing->book_history_id)
                ->update(['status' => 'Available']);
        });
    }
    public function libraryBookBorrowingRenew(Request $request)
    {
        $fiveDaysAgo = Carbon::today()->addDays(5);
        $data = BookBorrowing::find($request->id);
        if ($data->renewals >= 2) {
            return response()->json('Maximum renewal limit reached');
        } else {
            $data->renewals++;
            $data->due_date = $fiveDaysAgo;
            $data->save();
        }
    }
    public function getBorrowingBook()
    {
        $today = Carbon::today();

        $BookBorrowing = BookBorrowing::with(['member', 'book.bookAuthor', 'BookHistory'])->whereNull('returned_date')->paginate(15);

        $BookBorrowing->getCollection()->transform(function ($b) use ($today) {
            $dueDate = Carbon::parse($b->due_date);
            $overdueDays = $today->greaterThan($dueDate)
                ? $today->diffInDays($dueDate)
                : 0;
            // Determine status
            $status = $b->renewals > 0 ? 'renewed' : ($overdueDays > 0 ? 'overdue' : $b->status);

            $isOverdue = $today->greaterThan($dueDate);

            return [
                'id'            => $b->id,
                'status'        => $status,
                'isOverdue'     => $isOverdue,
                'memberName'    => $b->member->full_name ?? '',
                'memberEmail'   => $b->member->email ?? '',
                'memberPhone'   => $b->member->phone ?? '',
                'bookTitle'     => $b->book->title ?? '',
                'book_id'       => $b->book->id ?? '',
                'book_code'     => $b->BookHistory->book_code ?? '',
                'bookName'      => $b->book->book_name ?? '',
                'bookAuthor'    => $b->book->bookAuthor->name ?? '',
                'borrowed_date' => $b->borrowed_date,
                'due_date'      => $b->due_date,
                'renewals'      => $b->renewals,
            ];
        });
        return $BookBorrowing;
    }
    public function storeFineConfiguration(Request $request)
    {
        $fineDetails        = $request->input('fine_configuration');
        $site_id            =   session()->get('SITE_ID');
        $fine_configuration = LibraryFineConfiguration::where('site_id', $site_id)->first();

        if ($fine_configuration) {
            $fine_configuration->fine_details = $fineDetails;
            $fine_configuration->save();
        } else {
            LibraryFineConfiguration::create([
                'fine_details' => $fineDetails,
            ]);
        }
        return 'done';
    }

    public function getLibraryFine()
    {
        $site_id = session()->get('SITE_ID');

        $fine = LibraryFineConfiguration::where('site_id', $site_id)->first();

        if ($fine && is_array($fine->fine_details)) {
            return $fine->fine_details;
        }
        return [
            [
                'day' => 1,
                'amount' => 0.0,
            ]
        ];
    }
}
