<?php

namespace App\Http\Controllers\Edu\SitePayment;

use App\Models\Academic\AcademicOnlinePaymentMethod;
use App\Models\Edu\PaymentGateway\EduPaymentGateway;
use App\Models\Edu\SitePayment\EduInvoicePayment;
use App\Models\Edu\SitePayment\SiteInvoice;
use App\Models\Site\SiteInfo;
use App\Models\Site\SiteMembership;
use App\Traits\Invoice\InvoiceHistoryTrait;
use App\Traits\Site\Payment\SiteBkashTransactionTrait;
use Auth;
use Carbon\Carbon;
use DB;
use Error;
use Exception;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use App\Http\Controllers\Controller;
use Illuminate\Validation\ValidationException;
use PDF;
use Response;
use Session;

class SitePaymentController extends Controller
{


    use InvoiceHistoryTrait, SiteBkashTransactionTrait;

    private $invoicePayment;
    private $siteMembership;
    private $siteInfo;
    private $invoice;

    public function __construct(EduInvoicePayment $invoicePayment, SiteInvoice $invoice, SiteInfo $siteInfo, SiteMembership $siteMembership)
    {
        $this->middleware('auth');
        $this->middleware('eduworlduserchecker');
        $this->siteInfo = $siteInfo;
        $this->siteMembership = $siteMembership;
        $this->invoicePayment = $invoicePayment;
        $this->invoice = $invoice;

    }


    /**
     * @param Request $request
     * @return JsonResponse
     */
    public function createPayment(Request $request)
    {
//        return $request;


        try {
            $this->validate($request, [
                'invoice_id' => 'required|numeric|min:1',
                'account_code' => 'required',
                'payment_method' => 'required',
                'reference_no' => 'required_if:payment_method,bkash',
                'paid_amount' => 'required|numeric|min:1',
            ]);

            DB::beginTransaction();

            $current_date_time = Carbon::now();
            $current_date = $current_date_time->toDateString();

            $the_invoice = $this->invoice->findOrfail($request->invoice_id);

            //check invoice is carried or not
            if (strcasecmp($the_invoice->status_code, "carried") == 0) {
                $the_carried_to_invoice = $this->invoice->findOrfail($the_invoice->carried_invoice_id);
                throw new Error('The Invoice Is Carried To: ' . $the_carried_to_invoice->invoice_no);
            }


            if (strcasecmp($the_invoice->status_code, 'draft') == 0) {
                throw new Error('The Invoice: ' . $the_invoice->invoice_no . ' Is Not Approved By EduTeam');
            }

            if ($the_invoice->amount_due < $request->paid_amount) {
                throw new Error('The Invoice Due: ' . $the_invoice->amount_due . ' But you are going to pay: ' . $request->paid_amount);
            }


            $this->invoicePayment->invoice_id = $request->invoice_id;
            $this->invoicePayment->account_code = $request->account_code;
            $this->invoicePayment->payment_method = $request->payment_method;
            $this->invoicePayment->reference_no = $request->reference_no;
            $this->invoicePayment->paid_amount = $request->paid_amount;
            $this->invoicePayment->status_code = 'approved';
            $this->invoicePayment->payment_date = $request->payment_date ?? $current_date;
            $this->invoicePayment->site_id = $the_invoice->site_id;
            $this->invoicePayment->received_by = Auth::user()->id;
            $this->invoicePayment->save();

            $the_invoice = $this->invoice->findOrfail($request->invoice_id);

            /*
                if ($this->invoice->payment_type == 'sms') {
                    $site_membership = $this->siteMembership
                        ->where('site_id', $this->sitePayment->site_id)
                        ->first();
                    $site_membership->sms_balance = ((integer)$site_membership->sms_balance + (integer)$this->sitePayment->piece);
                    $site_membership->save();

                }*/


            $this->createHistory($the_invoice->id, 'approved', 'Invoice:' . $the_invoice->invoice_no . ' Payment Taken: ' . $request->paid_amount);

            $the_invoice->status_code = 'partial';
            $this->createHistory($the_invoice->id, 'draft', 'Invoice:' . $the_invoice->invoice_no . ' Status Changed to Partial');


            if ($the_invoice->amount_due == 0) {
                $the_invoice->status_code = 'paid';
                $this->createHistory($the_invoice->id, 'draft', 'Invoice:' . $the_invoice->invoice_no . ' Status Changed to Paid');
            }
            $the_invoice->save();


            DB::commit();
            return Response::json(['message' => 'Payment Created Successfully for invoice: ' . $the_invoice->invoice_no, 'mode' => 'Saved!'], 200);


        } catch (ValidationException $exception) {
            return JsonResponse::create(['message' => $exception->getMessage(), 'errors' => $exception->validator->getMessageBag()->toArray()], 422);
        } catch (Exception $exception) {
            DB::rollBack();
            return JsonResponse::create($exception->getMessage(), 500);

        }
    }


    public function getInvoicePayments($invoice_id)
    {
        try {
            $list = $this->invoicePayment
                ->where('invoice_id', $invoice_id)
                ->get();
            return response()->json($list, 200, []);
        } catch (Exception $exception) {
            return response()->json($exception->getMessage(), 411, []);

        }

    }

    public function getSitePayments(Request $request)
    {
        try {

            $status_codes = $request->status_codes;
            $search_txt = $request->search_txt;
            $paginate = $request->paginate ?? 20;
            $status = $request->status ?? 1;
            $date_range = $request->date_range ? json_decode($request->date_range) : null;
            $item_id = $request->item_id;

            $site_id = Session::get('SITE_ID');


            if (!$status_codes) {
                throw new Error('Check At Least One Status Code!');
            }


            $txt = '%' . $search_txt . '%';

            $list = $this->invoicePayment
                ->where('site_id', $site_id)
                ->with(['invoice'=>function($q){
                    $q->select(['id','invoice_no','grand_total']);
                }])
                ->whereIn('status_code', $status_codes)
                ->where('status', $status)
                ->when($search_txt, function ($q) use ($txt) {
                    return $q->whereHas('invoice', function ($q1) use ($txt) {
                        $q1->where('invoice_no', 'LIKE', $txt);
                    });
                })->when($item_id, function ($q) use ($item_id) {
                    return $q->whereHas('invoice', function ($q1) use ($item_id) {
                        $q1->whereHas('details',function ($q2) use ($item_id) {
                            $q2->where('item_details_id', $item_id);

                        });//->select('id');
                    });
                })->when($date_range, function ($q) use ($date_range) {
                    return $q->where(function ($query) use ($date_range) {
                        $query->whereBetween('payment_date', [$date_range->start, $date_range->end]);
                    });

                })
                ->orderBy('id', 'DESC')
               // ->get();
                ->paginate($paginate);



           // dd($list);



            $amount_paid = $list->sum('paid_amount');


            return response()->json([$list, $amount_paid], 200, []);
        } catch (Exception $exception) {
            return response()->json($exception->getMessage(), 411, []);

        }

    }


    //=====================================site bkash transaction for edu========================================
    public function eduPaymentMethodList()
    {
        $payment_method_keys = EduPaymentGateway::pluck('academic_payment_method_key');
        $payment_method_list = AcademicOnlinePaymentMethod::whereIn('name_key', $payment_method_keys)->get();
        return $payment_method_list;
    }

    public function paymentPage($invoice_no = null, $total_amount = null)
    {
        $viewType = 'Site Payment';

//        $the_invoice = $this->invoice->where('invoice_no', $invoice_no)->first();
//        if (strcasecmp($the_invoice->status_code, "carried") == 0) {
//            $the_carried_to_invoice = $this->invoice->findOrfail($the_invoice->carried_invoice_id);
//            throw new Error('The Invoice Is Carried To: ' . $the_carried_to_invoice->invoice_no);
//        }
//
//
//        if (strcasecmp($the_invoice->status_code, 'approved') == 0) {
//            throw new Error('The Invoice: ' . $the_invoice->invoice_no . ' Is Not Approved By EduTeam');
//        }

        return view('default.admin.site.site-payment.site-bkash-payment-for-edu', compact('viewType', 'invoice_no', 'total_amount'));
//        return view('default.admin.layouts.master', compact('viewType'));
    }

    public function getToken(){
//        return 'ok';
        $token = $this->token();

        return $token;
    }

    public function creatBkashPayment(){
        $create_payment = $this->createBkashEduPayment();
        return $create_payment;
    }

    public function executeBakashPayment(Request $request){
        $execute_payment = $this->executePayment($request);
        return $execute_payment;
    }
}
