<?php

namespace App\Console\Commands;

use App\Traits\DB\DatabaseBackupTrait;
use Carbon\Carbon;
use Illuminate\Console\Command;
use Illuminate\Support\Facades\Storage;  // এটা যোগ করো
use ZipArchive;
use Symfony\Component\Process\Process;
use Symfony\Component\Process\Exception\ProcessFailedException;

class BackupDatabase extends Command
{
    use DatabaseBackupTrait;

    protected $signature = 'db:backup';
    protected $description = 'Backup the database';
    protected $process;

    public function __construct()
    {
        parent::__construct();

        $date = Carbon::now()->format('Y-m-d');
        $file_path = 'backups/' . $date . '-backup.sql';

        $this->process = new Process(sprintf(
            'mysqldump -u%s -p%s -h%s %s > %s',
            config('database.connections.mysql.username'),
            config('database.connections.mysql.password'),
            config('database.connections.mysql.host'),
            config('database.connections.mysql.database'),
            storage_path($file_path)
        ));
    }

    public function handle()
    {
        try {
            $this->process->mustRun();
            $this->info('✅ SQL backup completed.');

            $backupDir = storage_path('backups');
            $date = Carbon::now()->format('Y-m-d');
            $sqlFile = "{$backupDir}/{$date}-backup.sql";
            $zipFile = "{$backupDir}/{$date}-backup.zip";

            // Zip the SQL file
            $zip = new ZipArchive();
            if ($zip->open($zipFile, ZipArchive::CREATE | ZipArchive::OVERWRITE) === TRUE) {
                $zip->addFile($sqlFile, basename($sqlFile));
                $zip->close();
                $this->info("📦 Zipped: {$zipFile}");

                // Delete original SQL after zipping
                if (file_exists($sqlFile)) {
                    unlink($sqlFile);
                    $this->info("🗑️ Deleted .sql: {$sqlFile}");
                }
            } else {
                $this->error('❌ Failed to zip backup file.');
                return; // zip failed, stop further execution
            }

            // Upload zip to Google Drive
        //    $this->uploadBackupToGoogleDrive();

            // Cleanup old zip files (older than 7 days)
            $this->cleanupOldZips();

        } catch (ProcessFailedException $e) {
            \Log::error($e->getMessage());
            $this->error('❌ Backup failed: ' . $e->getMessage());
        }
    }


    protected function cleanupOldZips()
    {
         $backupPath = storage_path('backups');

            if (!is_dir($backupPath)) {
                $this->info("ℹ️ Backup path not found: {$backupPath}");
                return;
            }

            $files = collect(scandir($backupPath))
                ->filter(function ($file) {
                    return !in_array($file, ['.', '..']) && substr($file, -4) === '.zip';
                })
                ->map(function ($file) use ($backupPath) {
                    $fullPath = $backupPath . DIRECTORY_SEPARATOR . $file;
                    return [
                        'name' => $file,
                        'path' => $fullPath,
                        'time' => file_exists($fullPath) ? filemtime($fullPath) : 0,
                    ];
                })
                ->sortByDesc('time') // নতুন ফাইল আগে
                ->values();

            if ($files->isEmpty()) {
                $this->info("ℹ️ No zip files found to clean.");
                return;
            }

            // সর্বশেষ 7টি রেখে বাকি ডিলিট করো
            $filesToDelete = $files->slice(7);

            foreach ($filesToDelete as $file) {
                if (file_exists($file['path'])) {
                    unlink($file['path']);
                    $this->info("🗑️ Deleted old zip: {$file['name']}");
                }
        }
    }

    public function uploadBackupToGoogleDrive()
    {
        $backupDir = storage_path('app/backups');
        $date = \Carbon\Carbon::now()->format('Y-m-d');
        $zipFile = "{$backupDir}/{$date}-backup.zip";
        $driveFileName = "{$date}-backup.zip";

        if (!file_exists($zipFile)) {
            $this->error("Backup file not found at $zipFile");
            return;
        }

        Storage::disk('google')->put($driveFileName, fopen($zipFile, 'r'));
        $this->info("✅ Uploaded to Google Drive: {$driveFileName}");
    }

    // public function uploadBackupToGoogleDrive()
    // {
    //     $backupDir = storage_path('backups');
    //     $date = Carbon::now()->format('Y-m-d');
    //     $zipFile = "{$backupDir}/{$date}-backup.zip";
    //     $driveFileName = "{$date}-backup.zip";

    //     if (!file_exists($zipFile)) {
    //         $this->error("Backup file not found at $zipFile");
    //         return;
    //     }

    //    // $contents = file_get_contents($zipFile);

    //     // Use Storage facade with google disk
    //   //  Storage::disk('google')->put($driveFileName, $contents);

    //     $stream = fopen($zipFile, 'r');
    //     Storage::disk('google')->put($driveFileName, $stream);
    //     fclose($stream);

    //     $this->info("Backup successfully uploaded to Google Drive as {$driveFileName}");
    // }
}
